/**
*
* \file        ethernet_cmds.c
* \brief       console commands routines and data for Ethernet/TCP/IP
*              interface.
* \author      Larry Salant
* \date        7/30/2008
* \note        Processor- and product-specific stuff should be kept out of here!
*              code adapted from AMS-AIP/TPS6
*/

////////////////////////////////////////////////////////////////////////////////

#include "ethernet.h"
#include "console.h"
#include "product.h"
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include "cip.h"
#include "gateway.h"
#include "hardware.h"
#include "string_utils.h"
#include "errors.h"
#include "os.h"
#include "nvl.h"
#include "DnsAutoDiscovery.h"
#include "dm_syscmds.h"
#ifdef STM32F2XX
#include "stm32f2xx_eth.h"
#endif

extern "C" UINT32 ENET_IsPHYLocal_Callback(void);
extern "C" UINT32 ENET_PHY_Address(void);
PTERLAPPBLE g_pSwitchSaveAppBLE;

////////////////////////////////////////////////////////////////////////////////
// Valid CIPIDs are 0x03 -> 0xFE.
#define MIN_CIP_ADDRESS 3
#define MAX_CIP_ADDRESS 0xfe

ETHER_PARAM InitialEtherParam;
ETHER_PARAM *pEtherParam = NULL;   // pointer to working copy for changes

#ifndef SERIES_3_CONTROL
const char *HostNameDefaultFormat ="%s-%02x%02x%02x";
#else
// Print in all uppercase for 3 series
const char *HostNameDefaultFormat ="%s-%02X%02X%02X";
#endif

#define MIN_TIME_TO_UPGRADE (60*9) // allow at least 9 minutes for upgrade

// Extended ethernet parameters for DNS support
#ifdef EXTENDED_ETHERNET_SUPPORT
EXTND_ETH_PARAM *pExtndEthParam = NULL;   // Pointer to working copy for changes
#endif

////////////////////////////////////////////////////////////////////////////////
/**
* \author    Larry Salant
* \brief     Initializes the ethernet parameters from NVL
* \date      7/30/2008
* \param     none
* \return    int
* \retval    0 == success or error
**/
INT32 InitEnetParams(void)
{
    UINT8 otp_mac[DM_MAC_ADDR_LEN];
    INT32 retVal;

    retVal = 0;

    pEtherParam = GetEnetParams();

#ifdef EXTENDED_ETHERNET_SUPPORT
    // Get extended ethernet parameters
    pExtndEthParam = GetExtndEthParams();
#endif

    // Don't initialize the ethernet if the MAC address from flash is bad
    if( !IsMACAddrValid(pEtherParam->MacAddr) )
    {
        // MNT - 2/5/2009 - Check to see if the OTP flash had the mac address in there
        // If so, just set the correct mac and force enable DHCP because IP addresses
        // might be invalid
        ReadMACAddrFromOTPFlash((char*)otp_mac);
        if( !IsMACAddrValid(otp_mac) )
        {
            // MAC address has never been programmed - throw an error
            DmSystemError(DM_ERROR_LEVEL_ERROR, DM_ERROR_SUBSYS_ETHERNET, ERR_ETH_MACADDRESS_INVALID,0);

            // we need to initialize the RAM copy of the ethernet parameters, anyway because if the DM switch
            // is in static IP mode it will keep rebooting the unit
            retVal = -1;

        } // if (memcmp(otp_mac, broadcast_mac, sizeof(broadcast_mac)) == 0)
        else
        {
            memcpy(pEtherParam->MacAddr, otp_mac, DM_MAC_ADDR_LEN);
            pEtherParam->UseDHCP = 1;
            TriggerSystemSave();
        }
    } // if (memcmp(pEtherParam->MacAddr, broadcast_mac, sizeof(broadcast_mac)) == 0)

    // make a copy so that people can update it without
    // breaking the operation
    memcpy(&InitialEtherParam,pEtherParam,sizeof(ETHER_PARAM));
    return retVal;
}

/**
* \author    Larry Salant
* \brief     parses ethernet addresses from the console
* \date      7/30/2008
* \param     name - string describing what is being set
* \param     pointer to struct to store value
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 Set_Address(char* name,unsigned long& param,char * cmd)
{
    // if no arguments
    if( cmd == 0 || !(*cmd) )                   // print current value
    {
        unsigned char* cp = (unsigned char*) &param;

#ifndef SERIES_3_CONTROL
        DmConsolePrintf("%s: %u.%u.%u.%u\r",name,cp[0],cp[1],cp[2],cp[3]);
#else
        // Print device 0 to match 3 series print; if more eth devices are needed, add later
        DmConsolePrintf("Device 0 %s: %u.%u.%u.%u\r",name,cp[0],cp[1],cp[2],cp[3]);
#endif
    }
    // if argument is a ?,
    else if( *cmd == '?' )                  // print help string
    {
        DmConsolePrintf("%s ip_address\r",name);
        return 0;
    }
    else // parse the command line
    {
        unsigned long u[4];
        char *p;
        char *p2;
        char *endptr;

        // convert dot decimal notation to 4 bytes
        // MNT - 3/3/2009 - First strip out any parameters - we only accept one parameter - the
        // address. strtok with a space will give the first parameter. Next, parse the IP address.
        p2 = strtok(cmd," ") ;
        p = strtok(p2,".") ;
        for( int iByte = 0; iByte < 4; iByte++ )
        {
            if( !p )
                return -1;
            u[iByte]  = strtol(p, &endptr, 10);
            if(( u[iByte] > 255 )||(*endptr != '\0'))
                return iByte + 1;
            // get next token
            p = strtok(NULL,".") ;
        }
        // concatenate the bytes
        param = (u[3] << 24) | (u[2] << 16) | (u[1] << 8) | u[0];
        DmConsolePrintf("%s saved. Reboot to take effect.\r",name);
        // save the setting in NVL
        TriggerSystemSave();
    }
    return 0;
}

/**
* \author    Larry Salant
* \brief     sets the IP Address of the device
* \date      7/30/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetIP_Address(UINT32 ignore, char * cmd)
{
#ifdef SERIES_3_CONTROL
  char *p = NULL;

  // Command help
  if ( (cmd != NULL) && (*cmd == '?') )
  {
    DmConsolePrintf("IPADDRESS [device_num ip_address]\r\n");
    DmConsolePrintf("\tdevice_num - specified Ethernet device  [0,0]\r\n");
    DmConsolePrintf("\tip_address - IP address in dot decimal notation\r\n");
    DmConsolePrintf("\tNo parameter - displays current value\r\n");
    return 0;
  }

  // Get first argument which is device number
  p = strtok(cmd," -,;:");

  // Get second argument which is the address string
  p = strtok(NULL," -,;:");

  return Set_Address("IP Address", pEtherParam->ip_address, p);
#else
  return Set_Address("IP Address",pEtherParam->ip_address,cmd);
#endif // #ifdef SERIES_3_CONTROL
}

/**
* \author    Larry Salant
* \brief     sets the subnet mask
* \date      7/30/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetSM_Address(UINT32 ignore, char * cmd)
{
#ifdef SERIES_3_CONTROL
  char *p = NULL;

  // Command help
  if ( (cmd != NULL) && (*cmd == '?') )
  {
    DmConsolePrintf("IPMASK [device_num ip_address]\r\n");
    DmConsolePrintf("\tdevice_num - specified Ethernet device  [0,0]\r\n");
    DmConsolePrintf("\tip_address - IP address in dot decimal notation\r\n");
    DmConsolePrintf("\tNo parameter - displays current value\r\n");
    return 0;
  }

  // Get first argument which is device number
  p = strtok(cmd," -,;:");

  // Get second argument which is the address string
  p = strtok(NULL," -,;:");

  return Set_Address("Subnet Mask", pEtherParam->subnet_mask, p);
#else
  return Set_Address("Subnet Mask",pEtherParam->subnet_mask,cmd);
#endif // #ifdef SERIES_3_CONTROL
}

/**
* \author    Larry Salant
* \brief     sets the default Router address
* \date      7/30/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetDR_Address(UINT32 ignore, char * cmd)
{
#ifdef SERIES_3_CONTROL
  char *p = NULL;

  // Command help
  if ( (cmd != NULL) && (*cmd == '?') )
  {
    DmConsolePrintf("DEFROUTER [device_num ip_address]\r\n");
    DmConsolePrintf("\tdevice_num - specified Ethernet device  [0,0]\r\n");
    DmConsolePrintf("\tip_address - IP address in dot decimal notation\r\n");
    DmConsolePrintf("\tNo parameter - displays current value\r\n");
    return 0;
  }

  // Get first argument which is device number
  p = strtok(cmd," -,;:");

  // Get second argument which is the address string
  p = strtok(NULL," -,;:");

  return Set_Address("Default Router", pEtherParam->def_router, p);
#else
  return Set_Address("Default Router",pEtherParam->def_router,cmd);
#endif // #ifdef SERIES_3_CONTROL
}

/**
* \author    Larry Salant
* \brief     sets the primary DNS address
* \date      7/30/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetPriDNSIP_Address(UINT32 ignore, char * cmd)
{
    return Set_Address("DNS Address",pEtherParam->pri_DNS,cmd);
}

#ifdef EXTENDED_ETHERNET_SUPPORT
/**
* \author    Larry Salant
* \brief     Sets the secondary DNS address
* \date      7/30/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetSecDNSIP_Address(UINT32 ignore, char * cmd)
{
    return Set_Address("Secondary DNS Address",pExtndEthParam->SecondDns,cmd);
}
#endif

/**
* \author    Larry Salant
* \brief     set the Host Name.
* \date      7/30/2008
* \param     sets the Host name for the DNS environment
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 EnterHostName(UINT32 ignore, char * cmd)
{
    // if no argument
    if( cmd == 0 || !(*cmd) )                   // no argument, print current value
    {
        DmConsolePrintf("Host Name: %s\r",pEtherParam->HostName[0] ? pEtherParam->HostName : "(none entered)");
    }
    else if( *cmd == '?' )                  // print help string
    {
        DmConsolePrintf("HOSTNAME [string | /CLEAR]\r");
        DmConsolePrintf("\tstring - ASCII string with host name\r");
        DmConsolePrintf("\t/CLEAR - clears the value\r");
    }

    else if( !strncmp(cmd,"/CLEAR",6) )
    {
        memset(pEtherParam->HostName,0,sizeof pEtherParam->HostName);
        DmConsolePrintf("Host name cleared. Reboot to take effect.\r");
        TriggerSystemSave();
    }
    else // parse the command line
    {
        UINT8 bLen = 0;
        for( char* cp = cmd; *cp && bLen < sizeof pEtherParam->HostName ; cp++ )
        {
            if( !isalnum(*cp) && *cp != '-' )
            {
                DmConsolePrintf("ERROR: Illegal character in host name.(A-Z,a-z,-)");
                return 0;
            }
			bLen++;
        }
        if( bLen >= sizeof pEtherParam->HostName )
        {
            DmConsolePrintf("ERROR: Host name cannot be > %d characters",sizeof pEtherParam->HostName);
            return 0;
        }

#ifdef SERIES_3_CONTROL
        // Force convert hostname to all upper case for 3 series
        LocalConvertEntireStringToUpper(cmd);
#endif

        // save the setting in NVL
        NetSetHostname(cmd);
        DmConsolePrintf("New host name set. Reboot to take effect.\r");
    }
    return 0;
}

/**
* \author    Larry Salant
* \brief     set the MAC Address.
* \date      7/31/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetMAC_Address(UINT32 ignore, char * cmd)
{
    UINT8 s[6];
    UINT8 i;

    // if no argument
    if( cmd == 0 || !(*cmd) )                   // no argument, print current value
    {
        RetrieveHWAddr(s);
        DmConsolePrintf("MAC address: %02x.%02x.%02x.%02x.%02x.%02x\r",s[0],s[1],s[2],s[3],s[4],s[5]);
    }
    else if( *cmd == '?' )                  // print help string
    {
        DmConsolePrintf("MACADDRESS mac address in dot hex notation\r");
    }
    else // parse the command line
    {
        unsigned long u[DM_MAC_ADDR_LEN];
        if( sscanf(cmd,"%02x.%02x.%02x.%02x.%02x.%02x",&u[0],&u[1],&u[2],&u[3],&u[4],&u[5]) != DM_MAC_ADDR_LEN )
            return 2;
        for( i = 0; i < 6; i++ )
            pEtherParam->MacAddr[i] = (unsigned char) u[i];
        // check if its a "valid" mac address
        if( !IsMACAddrValid(pEtherParam->MacAddr) )
            return 3;
        // set the Hostname accordingly.
        sprintf(pEtherParam->HostName,HostNameDefaultFormat,ProductGetName(),pEtherParam->MacAddr[3],pEtherParam->MacAddr[4],pEtherParam->MacAddr[5]);

        // save the setting in NVL
        TriggerSystemSave();

        // MNT - 2/5/2009 - Force Saving of MAC address from OTP flash
        SaveMACAddrInOTPFlash((char *)pEtherParam->MacAddr);

        DmConsolePrintf("MAC address saved. Reboot to take effect.\r");
    }
    return 0;
}
/**
* \author    Larry Salant
* \brief     saves the host name in NVL
* \date      7/31/2008
* \param     pointer to hostname string
* \return    void
* \retval    none
* \note      saves value in NVL, must reboot to take effect
**/
void NetSetHostname(char * p)
{
    _strncpy(pEtherParam->HostName,p, sizeof(pEtherParam->HostName));

    // save the setting in NVL
    TriggerSystemSave();
}

/**
  *
  * \author      RJen
  * \brief       Command for checking if the device has the ethernet stack enabled
  * \detail      Can only read the ethernet stack status for now
  * \date        12/21/2011
  * \param       argc - passed agruments if any
  * \param       cmd - pointer to the command string
  * \return      INT32
  * \retval      0 - if the command is successfull, -1 - if not
  *
  */
INT32 EthStackCmd(UINT32 ignore, char* cmd)
{
  // If no arguments, print current ethernet stack status
  if( !cmd || !(*cmd) )
    DmConsolePrintf("Ethernet Stack: On \r\n");

  // Show help for the DHCP command
  else if(*cmd == '?')
  {
    DmConsolePrintf("ETHERNET\r\n");\
      DmConsolePrintf("\tNo parameter - displays current setting\r\n");
  }

  return 0;
}

/**
* \author    Larry Salant
* \brief     enable/disable the use of DHCP for dynamic IP addressing
* \date      7/31/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 EnableDHCP(UINT32 ignore, char * cmd)
{
#ifndef EXTENDED_ETHERNET_SUPPORT
    // if no arguments or argument is a ?,
    if( !cmd || !(*cmd) || (*cmd == '?') )                   // print help string
    {
        DmConsolePrintf("DHCP [ON | OFF]\r");
        DmConsolePrintf("Current DHCP State: %s",((pEtherParam->UseDHCP == 0) ? "OFF": "ON"));
    }

// Extended ethernet includes support for more command arguments
#else
    // If no arguments, print current DHCP string
    if( !cmd || !(*cmd) )
    {
      // Display current DHCP state
      DmConsolePrintf("Current DHCP State: %s\r\n",((pEtherParam->UseDHCP == 0) ? "OFF": "ON"));
    }

    // Show help for the DHCP command
    else if (*cmd == '?')
    {
      DmConsolePrintf("DHCP [ON | OFF | REL_RENEW]\r\n");\
      DmConsolePrintf("\tON - enables DHCP for LAN A\r\n");
      DmConsolePrintf("\tOFF - disables DHCP for LAN A\r\n");
      DmConsolePrintf("\tREL_RENEW - performs a DHCP release and renew\r\n");
      DmConsolePrintf("\tNo parameter - displays current setting\r\n");
    }

    // Parse string argument for DHCP release and renew
    else if ( !stricmp(cmd,"REL_RENEW") )
    {
      // Release and renew
      if (IsDHCPEnabled())
      {
        int status = ReleaseDhcp();
        if (status)
        {
          DmConsolePrintf("ERROR: DHCP release failed.\r\n");
          return 0;
        }

        status = SetupDhcp();
        if (status)
        {
          DmConsolePrintf("ERROR: DHCP renew failed.\r\n");
          return 0;
        }
      } // if (IsDHCPEnabled())

      else
        DmConsolePrintf("ERROR: DHCP not enabled\r\n");
    } // else if ( !stricmp(cmd,"REL_RENEW") )
#endif // #ifndef EXTENDED_ETHERNET_SUPPORT

    else  // parse the command line
    {
        LocalConvertEntireStringToUpper(cmd);
        if( !strncmp(cmd,"ON", 2) )
            pEtherParam->UseDHCP = 1;
        else if( !strncmp(cmd,"OFF", 3) )
            pEtherParam->UseDHCP = 0;
        else // parse the command line
            return 1;
        DmConsolePrintf("New DHCP mode set. Reboot to take effect.\r\n");

#ifdef EXTENDED_ETHERNET_SUPPORT
        // Clear DNS servers if enabling DHCP
        if (pEtherParam->UseDHCP == 1)
        {
          pEtherParam->pri_DNS = 0;
          pExtndEthParam->SecondDns = 0;
        }
#endif

        // save the setting in NVL
        TriggerSystemSave();
    } // parse the command line
    return 0;
}
/**
* \author    Larry Salant
* \brief     if DHCP enabled, releases DHCP connection
* \date      7/31/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
**/
INT32 DhcpRelease(UINT32 ignore, char * cmd)
{
    int status;
    // if argument is a ?,
    if( cmd && *cmd == '?' )                   // print help string
    {
        DmConsolePrintf("RELEASE\r");
        DmConsolePrintf("\treleases the DHCP address\r");
    }
    else  // process the command
    {
        if( InitialEtherParam.UseDHCP )
        {
            status = ReleaseDhcp();
            if( status )
            {
                DmConsolePrintf("ERROR: DHCP release failed.\r");
                return 0;
            }
            else
                DmConsolePrintf("DHCP released.\r");

        }
        else
            DmConsolePrintf("ERROR: DHCP not enabled\r");
    }
    return 0;
}

/**
* \author    Larry Salant
* \brief     if DHCP enabled, renews DHCP connection
* \date      7/31/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 DhcpRenew(UINT32 ignore, char * cmd)
{
    int status;
    // if argument is a ?,
    if( cmd && *cmd == '?' )                    // print help string
    {
        DmConsolePrintf("RENEW\r");
        DmConsolePrintf("\trenews the DHCP address\r");
    }
    else  // process the command
    {
        if( InitialEtherParam.UseDHCP )
        {
            /*	D.R> 1/20/10
                Bugzilla 34970 - renew kils communication with devicde
                for some reason releasing address first will not allow it to be renewed
                requests go out to address 0.0.0.0
                need to look at it later, for now on just renew
                */
			/*
			D.R> 5/18/10
			Right now SetupDhcp is fixed, so it restarts DHCP process if thee address was released
			However renew should not release address before renewing, we have separate command for that (release)
			*/

#ifdef DHCP_RELEASE_BEFORE_RENEW
            status = ReleaseDhcp();
            if( status )
            {
                DmConsolePrintf("ERROR: DHCP release failed.\r");
                return 0;
            }
#endif // DHCP_RELEASE_BEFORE_RENEW

            status = SetupDhcp();
            if( status )
            {
                DmConsolePrintf("ERROR: DHCP renew failed.\r");
                return 0;
            }
            else
                DmConsolePrintf("DHCP renewing. Query IP address by issuing ESTATUS command\r");

        }
        else
            DmConsolePrintf("ERROR: DHCP not enabled\r");
    }
    return 0;
}
/**
* \author    Larry Salant
* \brief     for products that support standalone operation (e.g. Roombox)
*            adds an entry to the IP Table.
* \date      8/1/2008
* \param     pointer to command line arguments
*            cip_id ip_address/sitename
*              cip_id          - ID of the CIP node (in hex)
*              ip_address/name - IP address in dot decimal notation
*                                or name of the site for DNS lookup
*              port            - OPTIONAL CIP port
* \return    int
* \retval    1 = DHCP enabled
**/
INT32 AddMaster(UINT32 ignore, char * cmd)
{
    // if no arguments or argument is a ?,
    if( !cmd || !(*cmd) || (*cmd == '?') )                  // print help string
    {
        DmConsolePrintf("ADDMASTER cip_id ip_address/name device_id CIP_Port\r");
    }
    else // parse the command line
    {
        unsigned char  ip_addr[4];
        char*          pText;
        char*          SiteName;
        unsigned long  IP_Address;
        unsigned short deviceId = 0;
        unsigned short port = CIPGetPort();
        int            Flags;
        int            cnt;
        char* pError;     //Pointer to "strtoul" error
        unsigned long CIP_ID;

        //  read CIP_ID -ID of the CIP node (in hex)
        pText = strtok(cmd," ,");

        if (!pText)
            return -1;

        CIP_ID = strtoul(pText, &pError, 16);
        // report error if not a valid CIP ID. Valid CIPIDs are 0x03 -> 0xFE.
        if( *pError || CIP_ID < MIN_CIP_ADDRESS || CIP_ID > MAX_CIP_ADDRESS )
            return -1;

        //ip_address/name - IP address in dot decimal notation or name of the site for DNS lookup",
        // only look for spaces or commas so that '.' will be passed on the site name.
        SiteName = strtok(NULL," ,");

        if (!SiteName)
            return -1;

        // device id, if available, follows SiteName/IP address
        pText = strtok( NULL, " ,");
        if (pText)
        {
            deviceId = atoi(pText);

            // port, if available, follows device id
            pText= strtok( NULL, " ,");
            if (pText)
                port = atoi(pText);
        }

        // if site name given
        if( isalpha(*SiteName) )
        {
            // check for valid characters
            pText = SiteName;
            while( *pText )
            {
                if( !isalnum(*pText) && *pText != '-' && *pText != '.' )
                    return 2;
                pText++;
            }
            IP_Address = 0;
            Flags = CIP_STATIC_MASTER;
        }
        else // make sure its IP address in dot decimal notation
        {
            pText= strtok(SiteName," -.,;:");
            for( cnt = 0 ; cnt < 4 ; cnt++ )
            {
                if( pText )
                    ip_addr[cnt] = atoi(pText);
                else
                    break;
                pText = strtok(NULL," -.,;:") ;
            }
            if( cnt != 4 )
                return 2;
            IP_Address = *(unsigned long*) ip_addr;
            SiteName = NULL;
            Flags = CIP_STATIC_MASTER;
        }

#ifdef SERIES_3_CONTROL
        // Force convert sitename to all upper case for 3 series
        if (SiteName != NULL)
          LocalConvertEntireStringToUpper(SiteName);
#endif

        MASTER_LIST_ENTRY * pEntry = AddEntryToMasterList(CIP_ID,IP_Address,SiteName,deviceId,port,Flags);
        if( pEntry )
        {
            // MNT - 3/10/2009 - Fixed the printing of errors
            if( AddGatewayEntry(pEntry, pEntry->handle) != 0 )
                DmConsolePrintf("Reboot required.");
            else
                DmConsolePrintf("Master List stored. Reboot recommended\r");
        }
        else
        {
            DmConsolePrintf("Error: Could not add entry to master list.");
        }
    }
    return 0;
}



/**
* \author    Manish Talreja
* \brief     for products that support standalone operation (e.g. Roombox)
*            removes an entry to the IP Table.
* \date      8/14/2008
* \param     pointer to command line arguments
*            cip_id ip_address/sitename
*              cip_id          - ID of the CIP node (in hex)
*              ip_address/name - IP address in dot decimal notation
*                                or name of the site for DNS lookup
*              port            - OPTIONAL CIP port
* \return    int
* \retval    1 = DHCP enabled
**/
INT32 RemMaster(UINT32 ignore, char * cmd)
{
    // if no arguments or argument is a ?,
    if( !cmd || !(*cmd) || (*cmd == '?') )                  // print help string
    {
        DmConsolePrintf("REMMASTER cip_id ip_address/name\r");
    }
    else // parse the command line
    {
        unsigned char  ip_addr[4];
        char*          pText;
        char*          SiteName;
        unsigned long  IP_Address;
        int            cnt;
        char* pError;     //Pointer to "strtoul" error
        unsigned long CIP_ID;

        //  read CIP_ID -ID of the CIP node (in hex)
        pText = strtok(cmd," ,");

        if (!pText)
            return -1;

        CIP_ID = strtoul(pText, &pError, 16);
        // report error if not a valid CIP ID. Valid CIPIDs are 0x03 -> 0xFE.
        if( *pError || CIP_ID < MIN_CIP_ADDRESS || CIP_ID > MAX_CIP_ADDRESS )
            return -1;

        //ip_address/name - IP address in dot decimal notation or name of the site for DNS lookup",
        // only look for spaces or commas so that '.' will be passed on the site name.
        SiteName = strtok(NULL," ,");

        if (!SiteName)
            return -1;

        // if site name given
        if( isalpha(*SiteName) )
        {
            // check for valid characters
            pText = SiteName;
            while( *pText )
            {
                if( !isalnum(*pText) && *pText != '-' && *pText != '.' )
                    return 2;
                pText++;
            }
            IP_Address = 0;
        }
        else // make sure its IP address in dot decimal notation
        {
            pText= strtok(SiteName," -.,;:");
            for( cnt = 0 ; cnt < 4 ; cnt++ )
            {
                if( pText )
                    ip_addr[cnt] = atoi(pText);
                else
                    break;
                pText = strtok(NULL," -.,;:") ;
            }
            if( cnt != 4 )
                return 2;
            IP_Address = *(unsigned long*) ip_addr;
            SiteName = NULL;
        }

        int status = RemoveEntryFromMasterList(CIP_ID,IP_Address,SiteName);
        if( status==0 )
        {
            DmConsolePrintf("Master Removed. Reboot recommended\r");

            /// \todo Remove a gateway entry
#if 0
            if( AddGatewayEntry(pEntry, pEntry->handle) != 0 )
                DmConsolePrintf("AutoDiscovery: Error adding gateway entry");
#endif
        }

    }
    return 0;
}




/**
* \author    Larry Salant
* \brief     sets the CIP port number
* \date      8/8/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetCIPPort(UINT32 ignore, char * cmd)
{
    // if no argument
    if( cmd == 0 || !(*cmd) )                   // no argument, print current value
    {
        DmConsolePrintf("CIP port = %u\r",CIPGetPort());
    }
    else if( *cmd == '?' )                  // print help string
    {
        DmConsolePrintf("CIPPORT portnumber\r");
    }
    else // parse the command line
    {
        long PortNumber;
        char *p;

        // convert dot decimal notation to 4 bytes
        p = strtok(cmd," .") ;
        PortNumber = atoi(p);

        if( PortNumber <= 4096 )
        {
            DmConsolePrintf("Port number must be > 4096\r");
            return -1;
        }
        if( PortNumber == pEtherParam->CTP_PortNum )
        {
            DmConsolePrintf( "Port %u is already in use by CTP\r",PortNumber);
            return -1;
        }
        pEtherParam->CIP_PortNum = PortNumber;
        // save the setting in NVL
        TriggerSystemSave();
        DmConsolePrintf("New port number = %u. Reboot to take effect.\r",PortNumber);
    }
    return 0;
}
/**
* \author    Larry Salant
* \brief     sets the CTP port number
* \date      8/8/2008
* \param     pointer to command line arguments
* \return    int
* \retval    0 = OK or error
* \note      saves value in NVL, must reboot to take effect
**/
INT32 SetCTPPort(UINT32 ignore, char * cmd)
{
    // if no argument
    if( cmd == 0 || !(*cmd) )                   // no argument, print current value
    {
        DmConsolePrintf("CTP port = %u\r",CTPGetPort());
    }
    else if( *cmd == '?' )                  // print help string
    {
        DmConsolePrintf("CTPPORT portnumber\r");
    }
    else // parse the command line
    {
        long PortNumber;
        char *p;

        // convert dot decimal notation to 4 bytes
        p = strtok(cmd," .") ;
        PortNumber = atoi(p);

        if( PortNumber <= 4096 )
        {
            DmConsolePrintf("Port number must be > 4096\r");
            return -1;
        }
        if( PortNumber == pEtherParam->CIP_PortNum )
        {
            DmConsolePrintf( "Port %u is already in use by CIP\r",PortNumber);
            return -1;
        }
        pEtherParam->CTP_PortNum = PortNumber;
        // save the setting in NVL
        TriggerSystemSave();
        DmConsolePrintf("New port number = %u. Reboot to take effect.\r",PortNumber);
    }
    return 0;
}

/**
* \author    Larry Salant
* \brief     returns if DHCP is enable
* \date      7/31/2008
* \param     pointer to command line arguments
* \return    int
* \retval    1 = DHCP enabled
**/
UINT8 IsDHCPEnabled(void)
{
    return InitialEtherParam.UseDHCP;
}

/**
* \author    Larry Salant
* \brief     Retreives the hardware MAC address
* \date      7/30/2008
* \param     void
* \return    UINT32
* \retval    MAC address as an array of bytes
* \note      values loaded from NVL at poweron
**/
void RetrieveHWAddr (UINT8 *ether_addr)
{
    memcpy(ether_addr, InitialEtherParam.MacAddr,DM_MAC_ADDR_LEN);
}

/**
* \author    Larry Salant
* \brief     gets the CIP port number (what it was at boot)
* \date      8/8/2008
* \param     void
* \return    UINT32
* \retval    CIP Port number
* \note      values loaded from NVL at poweron
**/
UINT32 CIPGetPort(void)
{
    return pEtherParam->CIP_PortNum;
}

/**
* \author    Larry Salant
* \brief     gets the CTP port number (what it was at boot)
* \date      8/8/2008
* \param     void
* \return    UINT32
* \retval    CIP Port number
* \note      values loaded from NVL at poweron
**/
UINT32 CTPGetPort(void)
{
    return InitialEtherParam.CTP_PortNum;
}

/**
* \author    Larry Salant
* \brief     set ethernet defaults
* \date      7/30/2008
* \param     void
* \return    UINT32
* \retval    ip address (swapped)
* \note      values loaded from NVL at poweron
**/
void EthernetSetDefaults(void)
{
    unsigned char s[6];

    // make sure pointer has been initialized
#ifndef EXTENDED_ETHERNET_SUPPORT
    if( !pEtherParam )
#else
    if ( (!pEtherParam) || (!pExtndEthParam) )
#endif
        InitEnetParams();

    pEtherParam->ip_address    = 0;
    pEtherParam->subnet_mask   = 0x00FFFFFF;
    pEtherParam->def_router    = 0L;
    pEtherParam->Enabled       = true;
    pEtherParam->AutoNegotiate = AUTONEG_ON;
    pEtherParam->UseDHCP       = true;
    pEtherParam->pri_DNS       = 0;
    pEtherParam->CIP_PortNum   = 41794;
    pEtherParam->CTP_PortNum   = 41795;
    pEtherParam->AutodiscoveryEnabled = 1;

#ifdef EXTENDED_ETHERNET_SUPPORT
    // Set extended ethernet parameters to default
    pExtndEthParam->SecondDns = 0;
    pExtndEthParam->DhcpOptions = DHCP_OPTIONS_HOSTNAME;
    pExtndEthParam->NameResolutionMode = HOST_NAME_RESOLUTION_MODE_DNS;
    memset(pExtndEthParam->DomainName, 0, sizeof(pExtndEthParam->DomainName));
#endif

    // get the MAC address
    RetrieveHWAddr(s);
    // create the default hostname
    sprintf(pEtherParam->HostName,HostNameDefaultFormat,ProductGetName(),s[3],s[4],s[5]);

    // save the setting in NVL
    TriggerSystemSave();
}

/**
* \author    Larry Salant
* \brief     request the bootloader to upload new firmware via ethernet
* \date      1/15/2009
* \param     void
* \return    UINT32
* \retval    ip address (swapped)
**/
INT32 EthernetUpgrade(UINT32 ignore, char * cmd)
{
    UINT8 *pCmdBlock = GetBootCmdPtr();
//    AppBLoaderSharedData *pAppBLoaderSharedData;
//    UINT32 ipAddr;
//    UINT8 bDHCP;

#ifdef SERIES_3_CONTROL
    // Command help
    if (cmd && (*cmd == '?'))
    {
      DmConsolePrintf("IMGUPD - reboot into bootloader\r\n");
      return 0;
    }
#endif

#ifdef ETHERNET_SUPPORTED
    // check that we have a valid IP address before continuing
    UINT32 ipAddr = NetGetIPAddress();
#endif // ETHERNET_SUPPORTED
#ifdef APPLICATION_CLEANUP
    //	Note: to enable the application cleanup, set the APPLICATION_CLEANUP parameter in the Crestore IAR project file and define this function in the
	//	application. It was done this way not as a call or a pointer due to RAM and ROM space limitations in some DM cards.
	//	This function would prepare application cleanup such as memory release, stop processing or save eeprom parameters etc...
	ApplicationCleanup();
#endif
  //  if (ipAddr)
  {
    // make sure any non-volatile data is written to memory
    UpdateSystemNVL();

#ifdef ETHERNET_SUPPORTED
        UINT8 bDHCP = IsDHCPEnabled();
        // if DCHP is enabled and we our lease time is close to expiring, renew before rebooting.
        if( bDHCP && (NetGetDHCPLeaseTime() < MIN_TIME_TO_UPGRADE) )
        {
            DhcpRenew(ipAddr, (char *)&bDHCP);  // Arguments not really used.

            DmConsolePrintf("waiting for DHCP renewal\r");

            // delay to finish writing NVL and got new lease
            HwDelayMsec(2000);
        }
        else      // delay to finish writing NVL
            HwDelayMsec(500);
#endif // ETHERNET_SUPPORTED

        // turn off interrupts;  we will reboot after this.
        OsEnterCritical();

        // write the upgrade via ENET command at the beginning of RAM
        strcpy((char *)pCmdBlock, UPGRADE_FIRMWARE_ENET);

#ifdef ETHERNET_SUPPORTED
        // store the ip config information before the command
        // -RAM_EXCEPTION_TABLE_LENGTH is compensating for Exception table (two exception vectors)
        // doesn't matter if we overwrite RAM variables since we're about to reboot
        AppBLoaderSharedData *pAppBLoaderSharedData = (AppBLoaderSharedData*)(pCmdBlock - sizeof(AppBLoaderSharedData) -RAM_EXCEPTION_TABLE_LENGTH);

        // save IP address and DCHP state
        pAppBLoaderSharedData->currIPAddr = ipAddr;
        pAppBLoaderSharedData->isDHCPEnabled = bDHCP;

        // save console port
        pAppBLoaderSharedData->currCTPPort = CTPGetPort();

        // get the MAC address, host name, subnet mask and router
        RetrieveHWAddr(pAppBLoaderSharedData->MACAddr);
        NetGetHostname((char *)&pAppBLoaderSharedData->hostName[0], DM_HOST_NAME_LEN);
        pAppBLoaderSharedData->currSNMask = NetGetSubnetMask();
        pAppBLoaderSharedData->currDefRouter = NetGetDefRoute();

        // save hardware configuration
        pAppBLoaderSharedData->isPHYLocal = ENET_IsPHYLocal_Callback();
        pAppBLoaderSharedData->PHYAddr = ENET_PHY_Address();

        // any app that needs this off will do so in g_pSwitchSaveAppBLE below
        pAppBLoaderSharedData->bHDBaseTDMNETState = DM_HDBASET_DMNET_ENABLED_STATE;

#ifdef STM32F2XX
        
        // For passing warm signature to the next bootup
        if (stricmp(cmd,"WARM") == 0)
        {
            pAppBLoaderSharedData->warmReboot[0] = 'W';
            pAppBLoaderSharedData->warmReboot[1] = 'A';
            pAppBLoaderSharedData->warmReboot[2] = 'R';
            pAppBLoaderSharedData->warmReboot[3] = 'M'; 
        }
#endif

        if(g_pSwitchSaveAppBLE)
          (*g_pSwitchSaveAppBLE)(pAppBLoaderSharedData);

#endif // ETHERNET_SUPPORTED

        if( pfDmPrepareReboot )
        {
            pfDmPrepareReboot();
        }

        // reboot to come up in bootloader
        DmRebootNotFatal();
    }
    /*else
      DmConsolePrintf("Error - no ip address");*/

    return 0;
}

/**
* \author    Dariusz Rymsza
* \brief     request the bootloader to upload new firmware via non ethernet device
* \date      11/04/2010
* \param     void
* \return    UINT32
* \retval    none
**/
INT32 NonEthernetUpgrade(UINT32 ignore, char * cmd)
{
    UINT8 *pCmdBlock = GetBootCmdPtr();

    // make sure any non-volatile data is written to memory
    UpdateSystemNVL();

    // turn off interrupts;  we will reboot after this.
    OsEnterCritical();

    // write the upgrade via ENET command at the beginning of RAM
    strcpy((char *)pCmdBlock, UPGRADE_FIRMWARE_ENET);

    // reboot to come up in bootloader
    DmRebootNotFatal();

    return 0;
}

/**
* \author        Andrew Salmon

* \brief         Setup DHCP parameters for stand-alone mode if
*                previously in PNM mode
*
* \date          01/23/2012
* \details       Parses name string of connected card to find
*                out if TX/RX stand-alone mode. This is done
*                independently on both TX and RX cards. Enables
*                DHCP If DHCP is off an 169.254.x.x link local
*                address is found, then reboot self.
*
* \param         pCardName Connected card's name
* \param         pCardType this card type
*
* \return        none
*
* \retval        none
**/
void DMIpControllerStandaloneSetup( char *pCardName  , const char *pCardType)
{
    if( pCardType && pCardName)
    {
        // find the card Name starting at the first D in the ID String
        while ((*pCardName != 'D') && (*pCardName != '\0')) {
            pCardName++;
        }

        if(*pCardName != '\0')
        {
            //Check if this is an RX w/ TX controller
            if(strcmp(pCardType, "TX") == 0)
            {
                if((strncmp(pCardName, "DM-RMC", 6) == 0)||(strcmp(pCardName, "DM-FIBERTEST-RX") == 0))
                {
                    // check and reset VLAN state
                    if(( pEtherParam->UseDHCP == 0) && PNM_INTERNAL_NETWORK_HOST_ADDR(pEtherParam->ip_address))
                    {
                        pEtherParam->UseDHCP = 1;
                        TriggerCardSpecificSave();

                        RebootCmd(0, NULL);
                    }
                }
            }
        }
    }

}

/**
 *
 * \author      RJen
 * \brief       IP configuration command modeled after the same command on WINCE
 * \detail      Used for both "ipconfig" and "est"
 * \date        03/14/2013
 * \param       ignore - unused argument
 * \param       cmd - pointer to the command arguments
 * \return      INT32
 * \retval      0 - if the command is successfull, -1 - if not
 * \warning     none
 * \note        Make sure not to break toolbox while changing this command
 *
 */
INT32 IpConfigCmd(UINT32 ignore, char* cmd)
{
  char* pCmdStr = NULL;

   // Default print
  if ((cmd == NULL) || (*cmd == NULL))
  {
    DisplayEthernetSettings(DISPLAY_CURRENT_ENET_SETTINGS);
    return 0;
  }

  // Split the string into tokens to parse the agruments
  pCmdStr = strtok(cmd," :-/");

  // Safety check
  if (!pCmdStr)
    return 0;

  // Display full ethernet configuration
  if (strncasecmp(pCmdStr, "all", sizeof("all")-1) == 0)
  {
    DisplayEthernetSettings(DISPLAY_ADVANCED_ENET_SETTINGS);
  }
  // Release DHCP
  else if (strncasecmp(pCmdStr, "release", sizeof("release")-1) == 0)
  {
    ReleaseDhcp();

    // Only default adapter is supported currently
    DmConsolePrintf("Successfully Released adapter with index Number 0 \r\n");
  }
  // Renew DHCP
  else if (strncasecmp(pCmdStr, "renew", sizeof("renew")-1) == 0)
  {
    SetupDhcp();

    // Only default adapter is supported currently
    DmConsolePrintf("Successfully Renewed adapter with index Number 0 \r\n");
  }
  // Flush DNS
  else if (strncasecmp(pCmdStr, "flushdns", sizeof("flushdns")-1) == 0)
  {
    // Make sure name resolution interface is initialized before flushing
    if ( (g_pDnsAutoDiscovery != NULL) && g_pDnsAutoDiscovery->Initialized() )
    {
      g_pDnsAutoDiscovery->FlushDnsCache();
      DmConsolePrintf("Successfully flushed DNS cache \r\n");
    }
    else
      DmConsolePrintf("ERROR: DNS is not initialized yet, please retry later.\r\n");
  }
  // Stack stats
  else if (strncasecmp(pCmdStr, "stackstats", sizeof("stackstats")-1) == 0)
  {
    // Print statistics for LWIP
    PrintEthernetStatistics('T');
  }
  // MAC stats
  else if (strncasecmp(pCmdStr, "macstats", sizeof("macstats")-1) == 0)
  {
    PrintEthernetStatistics('C');
  }
  // For handling single character arguments
  else if (strnlen(pCmdStr, 1) == 1)
  {
    char SingleArg = toupper(*cmd);

    // For help menu
    if (SingleArg == '?')
      DisplayEthernetSettings(DISPLAY_HELP_ENET_SETTINGS);
    // For printing initial ethernet settings
    else if (SingleArg == 'I')
      DisplayEthernetSettings(DISPLAY_INITIAL_ENET_SETTINGS);
    // For other debug prints
    else
      PrintEthernetStatistics(SingleArg);
  }
  // Print help for any unknown arguments
  else
  {
    DisplayEthernetSettings(DISPLAY_HELP_ENET_SETTINGS);
  }

  return 0;
}

void SetDhcpTimeStamp( void )
{
    UINT8 *pTime;
    UINT32 time = HwGetMsec();

    pTime = (UINT8 *)&time;

    pEtherParam->DhcpTimeStamp[0] = *(pTime++);
    pEtherParam->DhcpTimeStamp[1] = *(pTime++);
    pEtherParam->DhcpTimeStamp[2] = *(pTime++);
    pEtherParam->DhcpTimeStamp[3] = *(pTime);

    // save the setting in NVL
    TriggerSystemSave();
}

#ifdef STM32F2XX

void InitMMCCounters(void)
{
  ETH_MMCResetOnReadCmd(ENABLE);
  ETH_MMCCounterRolloverCmd(ENABLE);
  ETH_MMCCounterFreezeCmd(DISABLE);

  // disable MMC interrupts
  ETH_MMCITConfig(ETH_MMC_IT_TGF, DISABLE);
  ETH_MMCITConfig(ETH_MMC_IT_TGFMSC, DISABLE);
  ETH_MMCITConfig(ETH_MMC_IT_TGFSC, DISABLE);
  ETH_MMCITConfig(ETH_MMC_IT_RGUF, DISABLE);
  ETH_MMCITConfig(ETH_MMC_IT_RFAE, DISABLE);
  ETH_MMCITConfig(ETH_MMC_IT_RFCE, DISABLE);

  ETH_MMCCountersReset();
}

void PrintMMCCounters(void)
{
  // transmitted good frames after a single collision
  // transmitted good frames after more than a single collision
  // transmitted good frames
  // received frames with CRC error
  // received frames with alignment error
  // received good unicast frames
  DmConsolePrintf("rx good %d\n", ETH_GetMMCRegister(ETH_MMCRGUFCR));
  DmConsolePrintf("tx good %d\n", ETH_GetMMCRegister(ETH_MMCTGFCR));
  DmConsolePrintf("rx CRC error %d\n", ETH_GetMMCRegister(ETH_MMCRFCECR));
  DmConsolePrintf("rx alignment error %d\n", ETH_GetMMCRegister(ETH_MMCRFAECR));
  DmConsolePrintf("tx single collision %d\n",   ETH_GetMMCRegister(ETH_MMCTGFSCCR));
  DmConsolePrintf("tx multiple collisions %d\n", ETH_GetMMCRegister(ETH_MMCTGFMSCCR));
}

#endif
